define(['angular', 'app', 'localResourceDirectoryService', 'authentication_service'], function (angular, app) {
	"use strict";
	app.service('fhirPatient', function ($http, $q, localResourceDirectoryService, authenticationService, patient) {

		var fhirPatientResources = {};
		var fhirPatient = {};
		var patientRefRequest = null;
		var patientIdReq = null;
		var patientFetchRquest = null;
		var forceFetchFhirPatient = false;

		// Fetch FHIR patient information based on the logged in user
		fhirPatientResources.fetch = function (forceFetch) {
			forceFetchFhirPatient = !!forceFetch;
			if (patientFetchRquest && !forceFetchFhirPatient) {
				return patientFetchRquest.promise;
			}

			var patientFetchPromise = $q.defer();

			patient.fetch().then(function(patient){
				resolvePatientByVaafiUserService(patient).then(function(response) {
					patientFetchPromise.resolve(response);
				});

				patientFetchRquest = patientFetchPromise;
			});

			return patientFetchPromise.promise;
		};

		// Resolve patient by internal FHIR ID reference, retries up to 5 times, once per second to wait for elastic indexing
		// Parameters:
		//  -reference: Patient/<FHIR unique id> or <FHIR unique id>
		var resolvePatientByReference = function (reference) {
			if(patientRefRequest) return patientRefRequest.promise;

			var patientResolvePromise = $q.defer();
			$q.all([localResourceDirectoryService.fetch(), authenticationService.checkAuthStatus()])
				.then(function (response) {
					if(response[1]) {
						var refId = reference.indexOf('/') >= 0 ? reference.split("/")[1] : reference;

						var retryCount = 0;

						var requestPatient = function(patient){
							$http.get(response[0]['patient'] + '/' + refId).success(function (response) {
								patientResolvePromise.resolve(response);
							}).success(function(response){
								if(!response.id){
									if(retryCount < 5){
										retryCount++;
										setTimeout(function(){ requestPatient(patient) }, 1000);
									}
									else{
										patientResolvePromise.reject("Expected patient not found");
										patientResolvePromise = null;
									}
								}
								else{
									patientResolvePromise.resolve(response);
								}
							}).error(function (error) {
								patientResolvePromise.reject(error);
								patientResolvePromise = null;
							});
						};

						requestPatient(reference);

						patientRefRequest = patientResolvePromise;
					}
				},
				function (error) {
					patientResolvePromise.reject(error);
					patientResolvePromise = null;
				}
			);

			return patientResolvePromise.promise;
		};

		// Resolve patient by internal VA identifier
		// If the FHIR-based patient record does not exist, it will be created using the necessary data points using known JWT payload
		// Parameters:
		//  -patient: response from User Service's patient object (in NextGen at /users/v1/session/)
		var resolvePatientByVaafiUserService = function (patient){
			if (patientIdReq && !forceFetchFhirPatient) return patientIdReq.promise;

			//determine unique ID and assigning authority based on available fields
			var assigningAuthority, uniqueId;
			if(patient['icn']){
				assigningAuthority = 'ICN';
				uniqueId = patient['icn'];
			}
			else if(patient['edipid']){
				assigningAuthority = 'EDIPI';
				uniqueId = patient['edipid'];
			}

			var lastName = patient.lastName;
			var firstName = patient.firstName;

			return requestFhirPatient(assigningAuthority, uniqueId, lastName, firstName);
		};

		function requestFhirPatient(assigningAuthority, uniqueId, lastName, firstName){
			var patientResolvePromise = $q.defer();
			$q.all([localResourceDirectoryService.fetch(), authenticationService.checkAuthStatus()])
				.then(function (response) {
					if(response[1]) {
						$http.get(response[0]['patient'] + '?identifier=' + uniqueId).success(function (patientResp) {
							if(!patientResp.total){
								//if no item returned then create a new reference with the supplied parameters and return that
								var baseUrl = response[0]['patient'];
								var system = response[0]['pgd-identifier-system'];
								createFhirPatient(baseUrl, system, assigningAuthority, uniqueId, lastName, firstName).success(function(response, staus, headers){
									//get response header's returned location
									var patientResource = headers()['content-location'];
									patientResource = patientResource.substring(patientResource.indexOf("Patient"));

									//now resolve patient by the content location
									patientResource = patientResource.split('/');
									resolvePatientByReference(patientResource[0] + '/' + patientResource[1]).then(function(response){
										patientResolvePromise.resolve(response);
									})
								});
							}
							else{
								patientResolvePromise.resolve(patientResp.entry[0].resource);
							}
						}).error(function (error) {
							patientResolvePromise.reject(error);
							patientResolvePromise = null;
						});
						patientIdReq = patientResolvePromise;
					}
				},
				function (error) {
					patientResolvePromise.reject(error);
					patientResolvePromise = null;
				}
			);

			return patientResolvePromise.promise;
		}

		var createFhirPatient = function (baseUrl, system, assigningAuthority, uniqueId, lastName, firstName) {
			return $http({
				method: 'POST',
				url: baseUrl,
				data: {
					"resourceType": "Patient",
					"identifier": [{
						"system": system,
						"type": {
							"coding": [{
								"code": assigningAuthority
							}]
						},
						"value": uniqueId
					}],
					"name": [{
						"use": "usual",
						"family": [ lastName ],
						"given": [ firstName ]
					}]
				}
			});
		};

		return fhirPatientResources;
	});
});